package automobile;

import java.util.Comparator;

/**
 * Gestion d'un véhicule
 * Contient sa plaque d'immatriculation, son compteur, sa jauge, la capacité de cette dernière et sa consommation
 * @author Adrien Poupa
 */
public class Vehicule {

    private static int registre = 0; // Variable partagée par toutes les instances de Véhicule
    private final int immatriculation;
    private Compteur compteur;
    private static final int capacite = 50; // Capacité du réservoir fixée à 50L
    private double jauge;
    private double consommationKilometrique;

    /**
     * Constructeur d'un Véhicule
     * Initialise sa consommation et sa plaque d'immatriculation
     * @param consommationKilometrique consommation en litre par 100 Km parcourus
     */
    public Vehicule(double consommationKilometrique) {
        // Initialisation de la consommation
        this.consommationKilometrique = consommationKilometrique;

        // Son immatriculation correspond à la valeur actuelle du registre, 0 si aucune classe n'a encore été instanciée
        immatriculation = registre;
        // Incrémentation du registre pour le possible futur nouveau véhicule
        registre++;

        // Initialisation du compteur
        compteur = new Compteur();
    }

    /**
     * Getter de jauge
     * @return jauge, le niveau de jauge actuel
     */
    public double getJauge() {
        return jauge;
    }

    /**
     * Getter du compteur
     * @return compteur, instance du compteur liée au véhicule actuel
     */
    public Compteur getCompteur() {
        return compteur;
    }

    /**
     * Getter du numéro d'immatriculation
     * @return immatriculation, numéro d'immatriculation du véhicule
     */
    public int getNoImmatriculation() {
        return immatriculation;
    }

    /**
     * Mettre de l'essence dans le véhicule
     * On prend en compte le niveau actuel de la jauge pour vérifier si on peut mettre l'intégralité de l'essence
     * @param litres litrage d'essence à mettre
     * @throws CapaciteDepasseeException si on tente de mettre plus d'essence que le réservoir ne peut en contenir
     */
    public void mettreDeLessence(int litres)
        throws CapaciteDepasseeException {
        // Si le total de ce qu'on veut mettre et ce qu'on a dépasse la capacité : erreur
        if (jauge+litres > capacite) {
            throw new CapaciteDepasseeException(litres);
        }
        else {
            // Sinon, on ajoute
            jauge += litres;
        }
    }

    /**
     * Faire le plein du véhicule
     * On met la jauge au maximum de la capacité du réservoir
     */
    public void faireLePlein() {
        jauge = capacite;
    }

    /**
     * On essaie de rouler le nombre de kilomètres spécifié
     * Si la jauge est vide, on fait le plein et on sort
     * Si la jauge permet de rouler le nombre de kilomètres désiré, aucun problème
     * Si on tente de rouler plus que la jauge ne le permet, on roule autant que possible et on fait le plein
     * @param kilometres nombre de kilomètres à rouler
     * @return nombre de kilomètres qu'on a pu parcourir
     */
    public double rouler(double kilometres) {
        // Si la jauge est vide, on ne peut rien parcourir et on fait le plein
        if (jauge == 0) {
            faireLePlein();
            return 0;
        }

        // Consommation pour le nombre de kilomètres donné
        double consommation = (consommationKilometrique*kilometres)/100;

        // Jauge restante une fois que les kilomètres seraient parcourus
        double jaugeRestante = jauge - consommation;

        // S'il resterait de l'essence,on parcourt
        if (jaugeRestante > 0) {
            compteur.add((int)kilometres); // ajout des kilomètres parcourus au compteur
            jauge = jaugeRestante;
            return kilometres;
        }
        // S'il resterait un total négatif, on veut parcourir le plus que l'on peut
        else {
            double kilometresPossibles = (100*jauge)/consommationKilometrique; // combien de kilomètres peut-on parcourir avec le réservoir?
            compteur.add((int)kilometresPossibles);  // ajout des kilomètres parcourus au compteur
            faireLePlein(); // on a vidé la jauge: on fait le plein
            return kilometresPossibles;
        }
    }

    /**
     * Surcharge toString pour l'affichage du véhicule
     * @return "Vehicule X : compteur = [Totalisateur = X | Partiel = X]; jauge = X"
     */
    @Override
    public String toString() {
        return "Vehicule " + immatriculation + " : compteur = " +
                "[" +"Totalisateur=" + compteur.getTotalisateur() + " | Partiel=" + compteur.getPartiel() +
                "]; jauge = " + jauge;
    }

    /**
     * Comparaison de deux véhicules en fonction de la plaque d'immatriculation
     * @param other autre véhicule à comparer à l'instance actuelle
     * @return -1 si l'autre a un numéro plus élevé, 1 si c'est l'inverse, 0 si les plaques sont égales
     * (ce qui ne devrait pas arriver)
     */
    public int compareTo(Vehicule other) {
        if (other.getNoImmatriculation() > getNoImmatriculation()) {
            return -1;
        }
        else if (other.getNoImmatriculation() < getNoImmatriculation()) {
            return 1;
        }
        return 0;
    }

    /**
     * Comparateur statique des compteurs
     */
    static class CompteurStaticComparator implements Comparator<Vehicule> {
        /**
         * Retour de l'instance statique
         * @return instance statique
         */
        public static CompteurStaticComparator getCompteurStaticComparator() {
            return new CompteurStaticComparator();
        }

        /**
         * Comparaison de deux voitures, tri par rapport à leur compteur
         * @param a Première voiture à tester
         * @param b Seconde voiture à tester
         * @return 1 si le compteur de a est supérieur au compteur de b, -1 s'il est inférieur, 0 si égal
         */
        @Override
        public int compare(Vehicule a, Vehicule b) {
            int compteurA = a.getCompteur().getTotalisateur(); // Compteur véhicule a
            int compteurB = b.getCompteur().getTotalisateur(); // Compteur véhicule b
            if (compteurA > compteurB) {
                return 1;
            }
            else if (compteurA < compteurB) {
                return -1;
            }
            return 0;
        }
    }

    class CompteurComparator implements Comparator<Vehicule> {
        /**
         * Comparaison de deux voitures, tri par rapport à leur compteur
         * @param a Première voiture à tester
         * @param b Seconde voiture à tester
         * @return 1 si le compteur de a est supérieur au compteur de b, -1 s'il est inférieur, 0 si égal
         */
        @Override
        public int compare(Vehicule a, Vehicule b) {
            int compteurA = a.getCompteur().getTotalisateur(); // Compteur véhicule a
            int compteurB = b.getCompteur().getTotalisateur(); // Compteur véhicule b
            if (compteurA > compteurB) {
                return 1;
            }
            else if (compteurA < compteurB) {
                return -1;
            }
            return 0;
        }
    }

}
